import arcpy
import os
from datetime import datetime
import sys

def add_fields_to_feature_classes(gdb_path):
    arcpy.env.workspace = gdb_path

    # Set up log file
    log_file = os.path.join(os.path.dirname(gdb_path), "field_addition_log.txt")

    # Tracking summary
    summary = {
        "feature_classes_found": 0,
        "feature_classes_skipped": 0,
        "fields_added": 0,
        "fields_skipped": 0
    }

    def log(message):
        timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        with open(log_file, "a") as f:
            f.write(f"[{timestamp}] {message}\n")
        print(message)

    # Clear previous log
    with open(log_file, "w") as f:
        f.write("=== Field Addition Log ===\n")

    # Dictionary of feature classes and their fields
    feature_classes = {
        "WBDHU8": [
            ("TNMID_Review", "TEXT", 300),
            ("HUC8_Review", "TEXT", 300),
            ("Name_Review", "TEXT", 300),
            ("RefGNISID_Review", "TEXT", 300)
        ],
        "WBDHU10": [
            ("TNMID_Review", "TEXT", 300),
            ("HUC10_Review", "TEXT", 300),
            ("Name_Review", "TEXT", 300),
            ("RefGNISID_Review", "TEXT", 300),
            ("HUType_Review", "TEXT", 300),
            ("Acres_Review", "TEXT", 300),
            ("AcresCalc", "DOUBLE")
        ],
        "WBDHU12": [
            ("TNMID_Review", "TEXT", 300),
            ("HUC12_Review", "TEXT", 300),
            ("Name_Review", "TEXT", 300),
            ("RefGNISID_Review", "TEXT", 300),
            ("HUType_Review", "TEXT", 300),
            ("HUMod_Review", "TEXT", 300),
            ("ToHUC_Review", "TEXT", 300),
            ("NonContribAcres_Review", "TEXT", 300),
            ("NonContribSqKm_Review", "TEXT", 300),
            ("Acres_Review", "TEXT", 300),
            ("AcresCalc", "DOUBLE"),
            ("Outlet_Review", "TEXT", 300)
        ],
        "WBDHU14": [
            ("TNMID_Review", "TEXT", 300),
            ("HUC14_Review", "TEXT", 300),
            ("Name_Review", "TEXT", 300),
            ("RefGNISID_Review", "TEXT", 300),
            ("HUType_Review", "TEXT", 300),
            ("HUMod_Review", "TEXT", 300),
            ("ToHUC_Review", "TEXT", 300),
            ("NonContribAcres_Review", "TEXT", 300),
            ("NonContribSqKm_Review", "TEXT", 300),
            ("Acres_Review", "TEXT", 300),
            ("AcresCalc", "DOUBLE"),
            ("Outlet_Review", "TEXT", 300)
        ],
        "WBDLine": [
            ("TNMID_Review", "TEXT", 300),
            ("HUDigit_Review", "TEXT", 300),
            ("Linesource_Review", "TEXT", 300),
            ("HUMod_Review", "TEXT", 300),
            ("Outlet_Review", "TEXT", 300),
            ("Geometry_Review", "TEXT", 300)
        ]
    }

    # Process each feature class
    for fc_name, fields in feature_classes.items():
        fc_path = os.path.join(gdb_path, fc_name)
        if arcpy.Exists(fc_path):
            log(f"Feature class '{fc_name}' found. Processing fields...")
            summary["feature_classes_found"] += 1
            existing_fields = [f.name for f in arcpy.ListFields(fc_path)]

            for field_name, field_type, *field_length in fields:
                if field_name not in existing_fields:
                    try:
                        if field_type == "TEXT" and field_length:
                            arcpy.AddField_management(fc_path, field_name, field_type, field_length=field_length[0])
                        else:
                            arcpy.AddField_management(fc_path, field_name, field_type)
                        log(f"  Added field: {field_name}")
                        summary["fields_added"] += 1
                    except Exception as e:
                        log(f"  ERROR adding field '{field_name}': {e}")
                else:
                    log(f"  Field '{field_name}' already exists. Skipping.")
                    summary["fields_skipped"] += 1
        else:
            log(f"Feature class '{fc_name}' not found. Skipping.")
            summary["feature_classes_skipped"] += 1

    # Write summary
    log("\n=== Summary ===")
    log(f"Feature classes found: {summary['feature_classes_found']}")
    log(f"Feature classes skipped: {summary['feature_classes_skipped']}")
    log(f"Fields added: {summary['fields_added']}")
    log(f"Fields skipped (already existed): {summary['fields_skipped']}")

if __name__ == "__main__":
    if len(sys.argv) > 1:
        gdb_path = sys.argv[1]
        add_fields_to_feature_classes(gdb_path)
    else:
        print("Usage: AddReviewFieldsTool.py <Path to Geodatabase>")
